# ------------------------------------------------------------------------------#
from typing import Any,Dict,Final,Tuple;
from datetime import datetime;
import logging,shutil,os;
import piexif;
from piexif import GPSIFD;

# ------------------------------------------------------------------------------#
logger=logging.getLogger('__name__');
sbr: str='\n\t\t  ';

# ------------------------------------------------------------------------------#
def to_rational(value: float, den: int = 1) -> Tuple[int, int]:
    # Konvertiert eine float-Zahl in ein Rational (Zähler, Nenner)
    num: int = int(round(value * den));
    return (num,den);
# end def to_rational

def get_alti_rational(alti: float, den:int=1):
    # den => Nenner der rationalen Zahl als Tupel
    isPos: bool= True if alti > 0.0 else False;
    talti=to_rational(alti,den);
    return (talti,isPos);
# end def get_alti_rational

def get_dezgrad_rational(deg: float,den: int=10_000):
    # den => Nenner der Sekunden als Rationaltupel
    isPos: bool= True if deg >= 0 else False;
    abs_d=abs(deg);
    d: int=int(abs_d);
    md: float=(abs_d - d)*60.0;
    m: int=int(md);
    sd: float=(md - m)*60.0;
    tpdec=((d,1),(m,1),to_rational(sd,den));
    return (tpdec,isPos);
# end def dezimalgrad_zu_gms

# ------------------------------------------------------------------------------#
ExifDict = Dict[str, Dict[int, Any]];

def apply_gps(exif_dict: ExifDict,locData) -> None:
    # Setzt/aktualisiert die GPS-Informationen in exif["GPS"].

    (Lat,Long,Alti)=locData;

    lat_dms,latIsPos=get_dezgrad_rational(Lat);
    lon_dms,longIsPos=get_dezgrad_rational(Long);
    alti_rat,altiIsPos=get_alti_rational(Alti);

    gps_lat_ref=('N' if latIsPos else 'S').encode("ascii");
    gps_long_ref=('E' if longIsPos else 'W').encode("ascii");
    gps_alti_ref=0 if altiIsPos else 1;
    # AltitudeRef: 0 = über NN, 1 = unter NN

    if "GPS" not in exif_dict: exif_dict["GPS"]={};
    gps_ifd=exif_dict.get("GPS",{});
    gps_ifd[GPSIFD.GPSLatitude]=lat_dms;
    gps_ifd[GPSIFD.GPSLatitudeRef]=gps_lat_ref;
    gps_ifd[GPSIFD.GPSLongitude]=lon_dms;
    gps_ifd[GPSIFD.GPSLongitudeRef]=gps_long_ref;
    gps_ifd[GPSIFD.GPSAltitude]=alti_rat;
    gps_ifd[GPSIFD.GPSAltitudeRef]=gps_alti_ref;
# end def apply_gps

# ------------------------------------------------------------------------------#
# ------------------------------------------------------------------------------#
EXIF_DATETIME_FORMAT: Final[str] = "%Y:%m:%d %H:%M:%S";

def format_exif_datetime(dt: datetime) -> str:
    # datetime -> EXIF-konforme String-Repräsentation.

    return dt.strftime(EXIF_DATETIME_FORMAT);
# end def format_exif_datetime

# ------------------------------------------------------------------------------#
def apply_time(exif: ExifDict, dt: datetime) -> None:
    exif["0th"][piexif.ImageIFD.DateTime] = format_exif_datetime(dt);
    exif["Exif"][piexif.ExifIFD.DateTimeOriginal] = format_exif_datetime(dt);
    exif["Exif"][piexif.ExifIFD.DateTimeDigitized] = format_exif_datetime(dt);
# end def apply_time

# ------------------------------------------------------------------------------#
# Dateizugriff / verlustfreies EXIF-Schreiben
# ------------------------------------------------------------------------------#
def read_jpeg_bytes(path: str) -> bytes:
    # Liest die JPEG-Bytes der Datei.
    with open(path, "rb") as f:
        return f.read();
# end def read_jpeg_bytes

# ------------------------------------------------------------------------------#
def load_exif(jpeg_bytes: bytes) -> ExifDict | None:
    # Lädt EXIF-Daten als Bytes
    try:
        exif_dict: ExifDict = piexif.load(jpeg_bytes);
        return exif_dict;
    except Exception:
        # Sollte nicht vorkommen, bereits voraus überprüft
        return None;
    # end try
# end def load_exif

# ------------------------------------------------------------------------------#
def write_exif_lossless(foto_path: str,exif: ExifDict):
    # Schreibt EXIF-Daten verlustfrei in eine JPEG-Datei.

    def to_backup_path(foto_path: str) -> str:
        return foto_path+'.orig';

    exif_bytes = piexif.dump(exif);

    backup_foto_path=to_backup_path(foto_path);
    # Nur falls das Backup noch nicht existiert!
    if not os.path.exists(backup_foto_path):
        shutil.copy2(foto_path,backup_foto_path);

    piexif.insert(exif_bytes,foto_path);
# end def write_exif_lossless

# ------------------------------------------------------------------------------#
# ------------------------------------------------------------------------------#
if __name__=='__main__':
    print('\n# Beginning jpg_exif_handling.py ...\n');
    # --------------------------------------------------------------------------#

    print('\n# Finished jpg_exif_handling.py.\n');
# end if main
# ------------------------------------------------------------------------------#
# ------------------------------------------------------------------------------#