# ------------------------------------------------------------------------------#
from typing import Dict,Tuple;
from datetime import datetime,timedelta;
import xml.etree.ElementTree as ET;
import nvector as nv;
# ------------------------------------------------------------------------------#
def get_datetime(s: str) -> datetime:
    return datetime.fromisoformat(s);
# end def
# ------------------------------------------------------------------------------#

type sCoords_Type=Tuple[str,str,str];
type fCoords_Type=Tuple[float,float,float];

def get_coords(elem: ET.Element,ns: Dict) -> sCoords_Type:
    lon=elem.get('lon');
    lat=elem.get('lat');
    ele=elem.find('GPX:ele',ns).text;
    return (lon,lat,ele);
# end def get_coords
# ------------------------------------------------------------------------------#

type Trk_sData_Type=Dict[str,sCoords_Type];
type Trk_fData_Type=Dict[datetime,sCoords_Type];

def extract_gpx_data(input_file_path:str)-> Tuple[str|None,Trk_fData_Type]:
    with open(input_file_path,mode='r',encoding='utf-8') as fReader:
        etree=ET.parse(fReader);
    root=etree.getroot();
    namesSpace=root.tag[1:root.tag.index('}')];
    ns={'GPX':namesSpace};
    # --------------------------------------------------------------------------#
    # Times with Points of track path
    trk=root.find('GPX:trk',ns);
    # Track name
    trkName=trk.find('GPX:name',ns).text;
    # Coordinates of track points
    trkseg=trk.find('GPX:trkseg',ns);
    trk_sdata: Trk_sData_Type={trkpt.find('GPX:time',ns).text:get_coords(trkpt,ns)
                               for trkpt in trkseg.findall('GPX:trkpt',ns)};
    trkfData: Trk_fData_Type={get_datetime(k):v for k,v in trk_sdata.items()};

    return (trkName,trkfData);
# end def
# ------------------------------------------------------------------------------#

def get_timeinterval(trD: Trk_fData_Type) -> Tuple[datetime,datetime]:
    lKeys=list(trD);
    lg=len(lKeys);
    return (lKeys[0],lKeys[lg-1])
# end get_timeinterval
# ------------------------------------------------------------------------------#

def extract_foto_locdata_from_gpxfile(trk_fdata,foto_dt: datetime):

    wpt_ftimes=sorted(list(trk_fdata.keys()));
    Lg=len(wpt_ftimes);

    time_interval: Tuple[datetime,datetime]=(wpt_ftimes[0],wpt_ftimes[0]);
    for ix in range(0,Lg-1):
        if (ix==0) and (foto_dt==wpt_ftimes[ix]):
            time_interval=(wpt_ftimes[ix],wpt_ftimes[ix]);
            break;
        # end if

        if (foto_dt <= wpt_ftimes[ix]):
            if (foto_dt == wpt_ftimes[ix]):
                time_interval=(wpt_ftimes[ix],wpt_ftimes[ix]);
                break;
            else:
                time_interval=(wpt_ftimes[ix-1],wpt_ftimes[ix]);
                break;
            # end if
        # end if
    # end for

    tp1_data=(time_interval[0],trk_fdata[time_interval[0]]);
    tp2_data=(time_interval[1],trk_fdata[time_interval[1]]);
    return(tp1_data,tp2_data);
# end def extract_foto_locdata_from_gpxfile
# ------------------------------------------------------------------------------#

def to_float(seq: sCoords_Type) -> tuple[float, ...]:
    return tuple(map(float,seq));

def determine_foto_locdata\
        (trackpt1,trackpt2,fotoTime: datetime)-> Tuple[float,...]:

    (tm_pt1,lc_pt1)=trackpt1;
    (tm_pt2,lc_pt2)=trackpt2;

    if lc_pt1==lc_pt2:
        (lon_foto,lat_foto,alti_foto)=to_float(lc_pt1);
        floc=(lat_foto,lon_foto,alti_foto);
        return floc;
    # end if

    delta_tm: timedelta=tm_pt2 - tm_pt1;
    assert (delta_tm!=0);
    fotoLocParam: float=(fotoTime - tm_pt1)/delta_tm;

    lon1,lat1,alt1=to_float(lc_pt1);
    lon2,lat2,alt2=to_float(lc_pt2);

    wgs84=nv.FrameE(name='WGS84');
    n_EB_E_t1=wgs84.GeoPoint(lat1,lon1,degrees=True).to_nvector();
    n_EB_E_t2=wgs84.GeoPoint(lat2,lon2,degrees=True).to_nvector();
    path12=nv.GeoPath(n_EB_E_t1,n_EB_E_t2);

    g_EB_E_foto=path12.interpolate(fotoLocParam).to_geo_point();
    lat_foto,lon_foto,z_foto=g_EB_E_foto.latlon_deg;

    # ------------------------------------------------------------------------------#
    # Interpoliere Höhendaten
    # Euklidischer (direkter) Abstand der Trackpunkte 1 und 2
    d_12=(n_EB_E_t2.to_ecef_vector()-n_EB_E_t1.to_ecef_vector()).length;
    # Euklidischer (direkter) Abstand des Trackpunktes 1 und und des Fotoortes
    d_1f=(g_EB_E_foto.to_ecef_vector()-n_EB_E_t1.to_ecef_vector()).length;

    diff_alti=alt2-alt1;
    assert (d_12!=0);
    factor=float(d_1f/d_12);
    alti_foto=factor*diff_alti+alt1
    # ------------------------------------------------------------------------------#
    lat_foto=float(lat_foto);
    lon_foto=float(lon_foto);
    floc=(lat_foto,lon_foto,alti_foto);
    return floc;
# end def determine_foto_locdata
# ------------------------------------------------------------------------------#
# ------------------------------------------------------------------------------#
if __name__=='__main__':
    print('\n# Beginning extract_fotolocdata_from_gpxfile.py ...\n');
    # --------------------------------------------------------------------------#

    print('\n# Finished extract_fotolocdata_from_gpxfile.py.\n');
# end if main
# ------------------------------------------------------------------------------#
# ------------------------------------------------------------------------------#