// #############################################################################

function y=fr2d(x),y=x*180.0/%pi,endfunction;
function y=fd2r(x),y=x*%pi/180.0,endfunction;

// #############################################################################

m1=1.0; // Pendelmasse in kg
l1=1.0; // Pendellänge in m

// Auslenkung der Masse
w1_0g=-90.0;
// w1_0g=1.0;

// Winkelgeschwindigkeit der Masse
w1dot_0g=254.0; // Scheitelpunkt: 253 kurz vor 180°, 254 kurz danach
//w1dot_0g=0.0;

// #############################################################################
// Simulationsdaten

t0=0;
t_end=20;       // Dauer in Sekunden
steptime=1/50;  // Bilder pro Sekunde

// #############################################################################

gc=9.81;

// #############################################################################

function rw1dot=fw1dot(w1,p1);
    rw1dot=p1;
endfunction;

function rp1dot=fp1dot(w1,p1)
    rp1dot=(-1)*(gc/l1)*sin(w1);
endfunction

function rekin=fekin(w1,p1)
    rekin=0.5*m1*l1^2*fw1dot(w1,p1)^2;
endfunction;

function rekindata=fekindata(ydata)
    for i=1:size(ydata,"c")
        w1=ydata(1,i);
        p1=ydata(2,i);
        rekindata(i)=fekin(w1,p1);
    end
endfunction;

function repot=fepot(w1)
    repot=(-1)*m1*gc*l1*cos(w1);
endfunction;

function repotdata=fepotdata(ydata)
    for i=1:size(ydata,"c")
        w1=ydata(1,i);
        p1=ydata(2,i);
        repotdata(i)=fepot(w1);
    end
endfunction;

function retot=fetot(w1,p1)
    retot=fekin(w1,p1)+fepot(w1)
endfunction

function retotdata=fetotdata(ydata)
    for i=1:size(ydata,"c")
        w1=ydata(1,i);
        p1=ydata(2,i);
        retotdata(i)=fetot(w1,p1);
    end
endfunction;

// #############################################################################

function rekin_kart=fekin_kart(w1,w1d)
    x1d=l1*cos(w1)*w1d;
    y1d=l1*sin(w1)*w1d;
    v1q=x1d**2+y1d**2;
    rekin_kart=0.5*m1*v1q;
endfunction;

function repot_kart=fepot_kart(w1)
    y1=-l1*cos(w1);
    repot_kart=m1*gc*y1;
endfunction;

// Berechnung der totalen Energie aus den Anfangsbedingungen über kartesische 
// Koordinaten - als Vergleichswert genutzt.
function retot_kart=fetot_kart(w1,w1d)
    retot_kart=fekin_kart(w1,w1d)+fepot_kart(w1);
endfunction;

// #############################################################################
// #############################################################################
// Anfangswerte

// t0=0;
w1_0=fd2r(w1_0g);
w1dot_0=fd2r(w1dot_0g);
x0=[w1_0;w1dot_0];

// #############################################################################

function xdot=fsp(t, x)
    w1=x(1);
    p1=x(2);
    xdot=fw1dot(w1,p1)*[1;0] ..
        +fp1dot(w1,p1)*[0;1];
endfunction;

// Simulationsdaten
steps=(t_end-t0)/steptime;
tdata=linspace(t0,t_end,steps);

odeType="rk";
// odeType="adams";
y=ode(odeType,x0,t0,tdata,fsp);

// #############################################################################
// Konstante der Bewegung

etotdata=fetotdata(y);
ekindata=fekindata(y);
epotdata=fepotdata(y);

etot0=etotdata(1); // Gesamtenergie, Konstante der Bewegung
px_etot=[t0;t_end];

delta_etot=etotdata-etot0;
delta_etot0=delta_etot(1);

funcprot(0);
// #############################################################################

px_null=[t0;t_end];

function fDisplayNull()
    null=0;
    py_null=[null;null];
    e=xpoly(px_null,py_null);
    e.foreground=color('grey');
end;

function fDisplayLine(yw)
    py_null=[yw;yw];
    e=xpoly(px_null,py_null);
    e.thickness=2;
    e.line_style=8;
    e.foreground=color('maroon');
endfunction;

// #############################################################################

function rng=fng(type)
    rng=type;
    if type=="" then
        rng="<not given>";
    end;
endfunction;

// #############################################################################

if %T then

    my_handle1=scf(1);
    clf(my_handle1,"reset");

    a=gca();
    a.x_label.text = "Zeit [s]";
    a.y_label.text = "Energien [Joule]";

    xgrid(color("lightgrey"));
    tEP='>   EP => '+string(w1_0g)+' | '+string(w1dot_0g) ..
                    +' || '+fng(odeType);
    xtitle(['Einfachpendel - Energieverhalten über die Zeit';tEP]);

    plot(tdata,etotdata,'red');
    plot(tdata,ekindata,'blue');
    plot(tdata,epotdata,'green');

    // py_etot=[etot0;etot0];
    // e=xpoly(px_etot,py_etot);
    // e.foreground=color('scilabmagenta3');

    legend(['Etot(t)';'Ekin(t)';'Epot(t)'],"lower_caption",%f);

    // -----------------------------------------------------------------------------

    my_handle2=scf(2);
    clf(my_handle2,"reset");
    a=gca();

    a.x_label.text = "Zeit [s]";
    a.y_label.text = "Abweichungen vom physikalischen Wert Etot  [Joule]";

    xgrid(color("lightgrey"));
    tEP='>   EP => '+string(w1_0g)+' | '+string(w1dot_0g) ..
                    +' || '+fng(odeType);
    xtitle(['Einfachpendel - Energieverhalten über die Zeit';tEP]);

    plot(tdata,delta_etot,'red');

    // py_etot=[delta_etot0;delta_etot0];
    // e=xpoly(px_etot,py_etot);
    // e.foreground=color('scilabmagenta3');

    legend(['Etot (t) - Etot(0)'],"lower_caption",%f);
end;

// #############################################################################

if %T then
    mean_etot=mean(etotdata);
    mean_ekin=mean(ekindata);
    mean_epot=mean(epotdata);
    mean_epot=mean(epotdata);

    max_etot=max(etotdata);
    max_ekin=max(ekindata);
    max_epot=max(epotdata);

    min_etot=min(etotdata);
    min_ekin=min(ekindata);
    min_epot=min(epotdata);

    init_etot=etotdata(1);
    init_ekin=ekindata(1);
    init_epot=epotdata(1);

    kart_ekin=fekin_kart(w1_0,w1dot_0);
    kart_epot=fepot_kart(w1_0);
    kart_etot=fetot_kart(w1_0,w1dot_0);

    max_Detot=max(delta_etot);
    mean_Detot=mean(delta_etot);
    min_Detot=min(delta_etot);

    mprintf('   \n');
    mprintf('---\n');
    mprintf('kart_ekin    : %g\n', kart_ekin);
    mprintf('init_ekin    : %g\n', init_ekin);
    mprintf('max_ekindata : %g\n', max_ekin);
    mprintf('mean_ekindata: %g\n', mean_ekin);
    mprintf('min_ekindata : %g\n', min_ekin);
    mprintf('---\n');
    mprintf('kart_epot    : %g\n', kart_epot);
    mprintf('init_epot    : %g\n', init_epot);
    mprintf('max_epotdata : %g\n', max_epot);
    mprintf('mean_epotdata: %g\n', mean_epot);
    mprintf('min_epotdata : %g\n', min_epot);
    mprintf('---\n');
    mprintf('kart_etot    : %g\n', kart_etot);
    mprintf('init_etot    : %g\n', init_etot);
    mprintf('max_etotdata : %g\n', max_etot);
    mprintf('mean_etotdata: %g\n', mean_etot);
    mprintf('min_etotdata : %g\n', min_etot);
    mprintf('---\n');
    mprintf('max_Detot    : %g\n', max_Detot);
    mprintf('mean_Detot   : %g\n', mean_Detot);
    mprintf('min_Detot    : %g\n', min_Detot);
    mprintf('---\n');
end;

// #############################################################################

if %F then // TBD
    p=polyfit(tdata,delta_etot,1);
    mprintf('p---\n');
    disp(p);
    r2d=180.0/%pi;
    mprintf('p(1)*r2d---\n');
    disp(p(1)*r2d);
    mprintf('p(2)---\n');
    disp(p(2));
end;

// #############################################################################

if %T then 

    function rw180data=fw180data(yL)
        rw180data=[];
        lg=size(yL,"c");
        for i=1:lg
            w=yL(1,i);
            vp=[cos(w);sin(w)];
            w=atand(vp(2),vp(1));
            rw180data(i)=w;
        end;
    endfunction;

    w180data=fw180data(y);
    max_w180data=max(w180data);

    // -----------------------------------------------------------------------------
 
    my_handle3=scf(3);
    clf(my_handle3,"reset");
    a=gca();
    a.x_label.text = "Zeit [s]";
    a.y_label.text = "Pendelwinkel [°]";

    xgrid(color("lightgrey"));
    tEP='>   EP => '+string(w1_0g)+' | '+string(w1dot_0g) ..
                     +' || '+fng(odeType);
    xtitle(['Einfachpendel - Pendelwinkel über die Zeit';tEP]);
    
    plot(tdata,w180data,'blue');
    legend(["$\theta(t)$"],"lower_caption",%f);
    fDisplayNull();
    if max_w180data > 170.0 then
        fDisplayLine(180);
        fDisplayLine(-180);
    end;
end;

// #############################################################################

disp('Beendet: circle_test_E-TOT-01.sce');