
from math import pi,sqrt,sin,cos,tan,atan,log;
#------------------------------------------------------------------------------#

# UTM / WGS84 Daten
Scale_On_Central_Meridian=0.9996;
Wgs84_Major_Axis=6378137.000;
Wgs84_Minor_Axis=6356752.3141;

#------------------------------------------------------------------------------#   

def mdist(Phi:float,E_Sq:float)->float:
    """ Meridian distance on the ellipsoid
    """
    A0=1.0-1.0*(E_Sq)/4.0-3.0*(E_Sq**2)/64.0-5.0*(E_Sq**3)/256.0-175.0*(E_Sq**4)/16384.0;
    A2=-3.0*(E_Sq)/8.0-3.0*(E_Sq**2)/32.0-45.0*(E_Sq**3)/1024.0-105.0*(E_Sq**4)/4096.0;
    A4=15.0*(E_Sq**2)/256.0+45.0*(E_Sq**3)/1024.0+525.0*(E_Sq**4)/16384.0;
    A6=-35.0*(E_Sq**3)/3072.0-175.0*(E_Sq**4)/12288.0;
    A8=315.0*(E_Sq**4)/131072.0;

    return(A0*Phi+A2*sin(2.0*Phi)+A4*sin(4.0*Phi)+A6*sin(6.0*Phi)+A8*sin(8.0*Phi));
# end mdist
#------------------------------------------------------------------------------#

def Lat_Long_To_Grid(Latitude:float,Longitude:float,Zone:int)->(int,int,float):
    """ Returns (Easting, Northing, Convergence) for Datum WGS84
    """
    Major_Axis=Wgs84_Major_Axis;
    Minor_Axis=Wgs84_Minor_Axis;
    Scaled_Major_Axis=Major_Axis*Scale_On_Central_Meridian;
    E_Squared=(Major_Axis**2-Minor_Axis**2)/Major_Axis**2;
    E_P_Squared=(Major_Axis**2-Minor_Axis**2)/Minor_Axis**2;

    Float_Northing=0.0; Float_Easting=0.0;

    False_Origin_East= 500000.0;
    False_Origin_North=0.0;

    Delta_Lambda=0.0;
    W0_Squared=0.0; W0=0.0;
    Ln=0.0;
    X=0.0; Y=0.0;
    C=0.0;

    Lambda_Ref_In_Degrees=abs(Zone)*6-183;
    Lambda_Ref= float(Lambda_Ref_In_Degrees)*(pi/180.0);

    Eta_0_Squared=0.0;

    if (Zone<0): False_Origin_North=10000000.0;
    else:        False_Origin_North=0.0;

    Delta_Lambda=Longitude-Lambda_Ref;

    W0_Squared=1.0-E_Squared*sin(Latitude)**2;
    W0=sqrt(W0_Squared);

    Ln=log((1.0+cos(Latitude)*sin(Delta_Lambda))/(1.0-cos(Latitude)*sin(Delta_Lambda)));
    X=(Scaled_Major_Axis/(2.0*W0))*Ln;
      
    Y=(Scaled_Major_Axis/W0)*(atan(tan(Latitude)/cos(Delta_Lambda))-Latitude);

    C=(W0_Squared*E_Squared*cos(Latitude)**2)/(6.0*(1.0-E_Squared)*Scaled_Major_Axis**2);

    Float_Easting=False_Origin_East+X-3.0*C*X*Y**2+C*X**3;
    Easting=int(Float_Easting);

    Float_Northing=False_Origin_North+Y+3.0*C*Y*X ** 2-C*Y ** 3+Scaled_Major_Axis*mdist(Latitude,E_Squared);
    Northing=int(Float_Northing);

    Eta_0_Squared=E_P_Squared*cos(Latitude)**2;
    Convergence=Delta_Lambda*sin(Latitude)+((Delta_Lambda**3)*sin(Latitude)*cos(Latitude)**2)*(1.0+3.0*Eta_0_Squared+2.0*Eta_0_Squared**2)/3.0;

    return(Easting,Northing,Convergence);
# Lat_Long_To_Grid;

#------------------------------------------------------------------------------#
#------------------------------------------------------------------------------#

if __name__ == '__main__':
    print('\n# Beginning main ...\n');

    r2d=180.0/pi;
    d2r=pi/180.0;

    # 'Matthias Claudius'
    MC_Long=(11.0+18.941/60.0)*d2r;
    MC_Lat=(48.0+7.200/60.0)*d2r;
    MC_Zone=32;

    MCU_Easting=672333;
    MCU_Northing=5332231;

    (MCC_Easting,MCC_Northing,MCC_Convergence)=Lat_Long_To_Grid(MC_Lat,MC_Long,MC_Zone);

    print(MCC_Easting-MCU_Easting,MCC_Northing-MCU_Northing,MCC_Convergence*r2d);

    print('\n# Finished main.\n');
# end if main

#------------------------------------------------------------------------------#
#------------------------------------------------------------------------------#