from typing import Tuple;
import csv,math,sys;
import datetime;
from enum import unique;

from geometry2d import *;
import angles;
from lat_long_to_grid import *;
from local_sphere import *;

# ------------------------------------------------------------------------------#

twopi=2.0*pi;
halfpi=pi/2.0;

rd2dg=180.0/pi;
dg2rd=pi/180.0;

# ------------------------------------------------------------------------------#

# dot to comma
def d2c(s: 'str') -> 'str':
    return s.replace('.',',');

# comma to dot
def c2d(s: 'str') -> 'str':
    return s.replace(',','.');

# ------------------------------------------------------------------------------#
# ------------------------------------------------------------------------------#
# Eingabedatei mit den Gipfel- und Ortsdaten
#
# Ortsdaten in der Form je Ort:
# " ,Ort, Länge Ost, Breite Nord, Höhe m,"
#
# Gipfeldaten in der Form je Gipfel:
# "Gebirgszug, Gipfelname, Länge Ost, Breite Nord, Höhe m, Ort"
#
# Länge und Breite sind die geografische Länge oder Breite in Bogengrad mit dem
# Punkt als Dezimaltrennungszeichen, die Höhe in Metern bezieht sich auf NN,
# der Ort bei den Gipfeldaten nennt einen bekannteren Ort in Gipfelnähe zur
# besseren Orientierung bei unbekannteren Gipfeln (oder auch mehrdeutigen
# Gipfelnamen).
# Die Koordinaten der Berggipfel sind zumeist keine 'amtlichen' Werte, ich habe
# diese aber immerhin :-) per Hand und Klick dem amtlichen elektronischen
# BayernAtlas (https://geoportal.bayern.de/bayernatlas/) entnommen. Auf ein paar
# Meter kommt es ja auch nicht an.
# ------------------------------------------------------------------------------#

inputFile="AlpenGipfel-uc-02.txt";

#
# Zeichenkodierung UTF-8 (ohne BOM)

# ------------------------------------------------------------------------------#
# Beobachtungsstandorte
# ------------------------------------------------------------------------------#

@unique
class BO_Orte(Enum):
    aubing=7;  # München Aubing (Imkerweg, Lärmschutzwall)
    dachau=3;  # Dachau (Schlosshof)
    eching=10;  # Eching bei Landshut (Friedhof)
    freiham=8;  # München Freiham (Schloss, Streiflacher Weg)
    germannsberg=9;  # Alling Germannsberg (Hochweg, Oberes Buchholz)
    gilching=12;  # Gilching Steinberg (Unterhalb Am Rinnerhof 1)
    herrsching=4;  # Herrsching (Ammersee, Kurpark)
    kerschlach=1;  # Pähl Kerschlach (Anhöhe)
    puchheim_ort=6;  # Puchheim Ort (Parsberg, Kreuthweg)
    sondelfingen=11;  # Reutlingen Sondelfingen (Achalm)
    starnberg=5;  # Starnberg (Würmsee, Dampfersteg)
    tutzing=2;  # Tutzing (Ilkahöhe)

# end class

# ------------------------------------------------------------------------------#

# Beobachtungsstandort >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
#
Mein_BO_Ort=BO_Orte.gilching;
#
# Eingabe erforderlich <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<

# ------------------------------------------------------------------------------#
# Beobachtungsstandorte - Daten
# ------------------------------------------------------------------------------#

# Hinweis: Die geografischen Längen und Breiten können in der folgenden Form
#   11,20433° Ost 47,92212° Nord
# (ohne mein Trennzeichen, das Semikolon ;) in des Suchfeld des BayernAtlas
# (https://geoportal.bayern.de/bayernatlas/) eingegeben werden

# Datenquelle für die Missweisungsdaten (Stand Oktober 2020):
# http://www-app3.gfz-potsdam.de/Declinationcalc/declinationcalc.html
# http://isdc.gfz-potsdam.de/igrf-declination-calculator/

if Mein_BO_Ort==BO_Orte.kerschlach:
    ### Beobachtungsstandort Kerschlach
    BO_Name='Pähl Kerschlach (Anhöhe)';
    BO_String="11,20433° Ost; 47,92212° Nord; 722 m";
    BO_Missweisung="3° 33'; 9.7 arcmin/year";
elif Mein_BO_Ort==BO_Orte.tutzing:
    ### Beobachtungsstandort Tutzing
    BO_Name='Tutzing (Ilkahöhe)';
    BO_String="11,25243° Ost; 47,90400° Nord; 703 m";
    BO_Missweisung="3° 34'; 9.7 arcmin/year";
elif Mein_BO_Ort==BO_Orte.dachau:
    ### Beobachtungsstandort Dachau
    BO_Name='Dachau (Hofgarten des Schlosses, Laubengang)';
    BO_String="11,431799° Ost; 48,257096° Nord; 510 m";
    BO_Missweisung="3° 37'; 9.8 arcmin/year";
elif Mein_BO_Ort==BO_Orte.herrsching:
    ### Beobachtungsstandort Herrsching
    BO_Name='Herrsching (Ammersee, Kurpark)';
    BO_String="11,16641° Ost; 47,99600° Nord; 533 m";
    BO_Missweisung="3° 33'; 9.7 arcmin/year";
elif Mein_BO_Ort==BO_Orte.starnberg:
    ### Beobachtungsstandort Starnberg
    BO_Name='Starnberg (Würmsee, Dampfersteg)';
    BO_String="11,34309° Ost; 47,99553° Nord; 584 m";
    BO_Missweisung="3° 36'; 9.7 arcmin/year";
elif Mein_BO_Ort==BO_Orte.puchheim_ort:
    ### Beobachtungsstandort Puchheim Ort
    BO_Name='Puchheim Ort (Parsberg, Kreuthweg)';
    BO_String="11,34343° Ost; 48,14316° Nord; 555 m";
    BO_Missweisung="3° 36'; 9.8 arcmin/year";
elif Mein_BO_Ort==BO_Orte.aubing:
    ### Beobachtungsstandort München Aubing
    BO_Name='München Aubing (Imkerweg, Lärmschutzwall)';
    BO_String="11,40730° Ost; 48,16212° Nord; 521 m";
    BO_Missweisung="3° 37'; 9.8 arcmin/year";
elif Mein_BO_Ort==BO_Orte.freiham:
    ### Beobachtungsstandort München Freiham
    BO_Name='München Freiham (Schloss, Streiflacher Weg)';
    BO_String="11,40358° Ost; 48,13539° Nord; 539 m";
    BO_Missweisung="3° 37'; 9.7 arcmin/year";
elif Mein_BO_Ort==BO_Orte.germannsberg:
    ### Beobachtungsstandort Germannsberg
    BO_Name='Alling Germannsberg (Hochweg, Oberes Buchholz)';
    BO_String="11,27798° Ost; 48,14349° Nord; 594 m";
    BO_Missweisung="3° 35'; 9.8 arcmin/year";
elif Mein_BO_Ort==BO_Orte.eching:
    ### Beobachtungsstandort Eching bei Landshut
    BO_Name='Eching bei Landshut (Friedhof)';
    BO_String="12,05723° Ost; 48,50443° Nord; 404 m";
    BO_Missweisung="3° 47'; 9.8 arcmin/year";
elif Mein_BO_Ort==BO_Orte.sondelfingen:
    ### Beobachtungsstandort Reutlingen Sondelfingen
    BO_Name='Reutlingen Sondelfingen (Achalm)';
    BO_String="9,24408° Ost; 48,49394° Nord; 707 m";
    BO_Missweisung="3° 6'; 10.1 arcmin/year";
elif Mein_BO_Ort==BO_Orte.gilching:
    ### Beobachtungsstandort Gilching Argelsried: Am Rinnerhof
    BO_Name='Gilching Steinberg (Unterhalb Am Rinnerhof 1)';
    BO_String="11,31067° Ost; 48,11124° Nord; 589 m";
    BO_Missweisung="3° 35'; 9.8 arcmin/year";
else:
    print("!!! Eingabedaten zu Mein_BO_Ort nicht definiert - Abbruch !!!");
    sys.exit();
# end if

# ------------------------------------------------------------------------------#

BO_Zone=32;  # UTM Zone ALLER Orte

# Der deutschsprachige Raum liegt größtenteils in den Zonen 32
# (6° bis 12° # östliche Länge) und 33 (12° bis 18° ö.L.).
#
# Für die Algorithmen müssen alle relevanten Orte in der UTM-Zone 32 liegen,
# wobei es (schmale) Überlappungsbereiche zwischen benachbarten UTM-Zonen
# gibt, in denen Orte UTM-Koordinaten aus beiden Zonen zugeordnet sind.
# Lässt sich ein Ort aus der 33iger Zone nicht für die 32iger Zone berechnen,
# wird das Python-Skript mit einer Fehlermeldung beendet.

# ------------------------------------------------------------------------------#
# ------------------------------------------------------------------------------#

def getMagneticDeviation(mwStr: str) -> Tuple[bool,float]:
    """
    :param mwStr: Vorgegebene Missweisungsdaten
    :return: (Ungültigkeit des Jahres, Nadelabweichung in Grad)
    """
    # Ist keine Nadelabweichung für ein Beobachtungsjahr angegeben, werden keine
    # Winkelwerte bezogen auf magnetisch Nord ausgegeben
    # Die Missweisungen sind gültig bis einschließlich zum 2025

    d1=datetime.date(2020,11,1)
    d2=datetime.date.today();

    if d2.year>2025:
        # Missweisungswerte nicht mehr gültig
        return (False,0.0);
    else:
        tmp=mwStr.split(';');
        tmp1=tmp[0].replace('°','').replace("'","");
        tmp1=tmp1.split(' ');
        mw20: float=float(tmp1[0])+float(tmp1[1])/60.0;
        dmw: float=float(tmp[1].replace('arcmin/year',''))/60.0;

        dm=(d2.year-d1.year)*12+d2.month-d1.month;
        mw: float=mw20+dmw*dm;
        #
        return (True,mw);
    # end if

# end getMagneticDeviation
#
# https://isdc.gfz-potsdam.de/igrf-declination-calculator/

# ------------------------------------------------------------------------------#
# ------------------------------------------------------------------------------#

tmp=BO_String.replace('°',' ').split(';');
bo_geod_long=float(c2d(tmp[0].split()[0]))*dg2rd;
bo_geod_lat=float(c2d(tmp[1].split()[0]))*dg2rd;
bo_alti=int(tmp[2].split()[0]);

# ------------------------------------------------------------------------------#

# Approximation des Ellipsoids am Beobachtungspunkt durch eine Kugelsphäre
# mit dem Gaußshen Krümmungsradius, des einfacheren Rechnens wegen
#

re_bo=radiusOfCurvature_Gauss(bo_geod_lat);

diff_NHN_Ellip=46.0;  # Meter
re_bo_NN=re_bo+diff_NHN_Ellip;

#
# diff_NHN_Ellip: Höhe des Geoids Normalhöhennull (NHN) über dem Ellipsoid
# WGS84 im Voralpenland um München herum;
# ich nehme an, dass die hier benutzte Kugelfläche das Ellipsoid um den
# Beobachtungspunkt 'hinreichend' genau approximiert.
#
# Keine Änderung der berechneten Ergebnisse gegenüber dem mittleren
# Erdradius re_84_NN !!!

# ------------------------------------------------------------------------------#

def getInputData(fName) -> List:
    with open(fName,'r',newline='',encoding='utf8') as fp:
        Reader=csv.reader(fp,delimiter=",");
        dataList=[];
        try:
            for row in Reader:
                gebiet=row[0].strip();
                name=row[1].strip();
                lat=float(row[2].strip())*dg2rd;
                long=float(row[3].strip())*dg2rd;
                alti_NN=int(row[4].strip());
                ort=row[5].strip();
                dataList.append((gebiet,name,long,lat,alti_NN,ort));
                #                   0     1     2     3     4       5
            # end for
        except csv.Error as Ex:
            sys.exit('file {}, line {}: {}'.format(inputFile,Reader.line_num,Ex));
        # end try
    # end with
    return dataList;

# end getInputData

# ------------------------------------------------------------------------------#

h_bo: float=bo_alti;  # m, Beobachterhöhe über NHN
# h_bo: float=0.0; # >>>TEMP-ONLY<<<
# Kartesisches Koordinatensystem im Mittelpunkt der Gaußschen Kugelsphäre

# h_bo: Höhe des Beobachterpunkts über NHN
# re_bo_NN: Radius der erweiterten Gaußschen Kugelsphäre in Höhe NHN
# d_bo_tg: Abstand des Beobachterpunkts vom Tangentenpunkt
#   Berechnet über das rechtwinkelige Dreieck
#       Tangentenpunkt-Kugelmittelpunkt-Beobachterpunkt
#   Gültig für alle Gipfelazimute
#
d_bo_tg: float=sqrt(h_bo*(2.0*re_bo_NN+h_bo));  # m

# Winkel Tangentenpunkt-Kugelmittelpunkt-Beobachterpunkt
beta: float=atan(d_bo_tg/re_bo_NN);  # Radian, 0<=beta<pi/2

# Kartesisches Koordinatensystem (x,y) in der Ebene aus dem Mittelpunkt
# der Gaußschen Kugelsphäre, dem Beobachterpunkt und dem Tangentenpunkt,
# so gewählt, dass der Beobachterpunkt auf der y-Achse liegt
#
# Kartesische Koordinaten des Beobachterpunktes in Höhe h_bo über NHN
bo_kt_x: float=0.0;
bo_kt_y: float=re_bo_NN+h_bo;

# Beobachtungspunkt in Höhe h_bo in kartesischen Kooedinaten
pt_kt_bo=Points(bo_kt_x,bo_kt_y);

# Kartesische Koordinaten des Tangentenpunktes
tg_kt_x: float=re_bo_NN*sin(beta);
tg_kt_y: float=re_bo_NN*cos(beta);

# Tangentenpunkt in kartesischen Koordinaten
pt_kt_tg=Points(tg_kt_x,tg_kt_y);

# Vektor vom Beobachtungspunkt in Höhe h_bo zum Tangentenpunkt
vc_kt_bo_tg=(Vectors(1.0,0.0) if pt_kt_tg==pt_kt_bo else pt_kt_tg-pt_kt_bo);

# ------------------------------------------------------------------------------#

# UTM-Koordinaten werden hier nur benutzt, um Abstände zu berechnen.
# Es ginge natürlich auch anders, also ohne die Umrechnung in UTM-Koordinaten.
# Beobachterpunkt: Umrechnung der geodötischen Länge und Breite
# in UTM Koordinaten
(bo_gd_east,bo_gd_north,BO_MConv)=Lat_Long_To_Grid(bo_geod_lat,bo_geod_long,BO_Zone);

# Beobachtungsfußpunkt in UTM-Koordinaten
pt_gd_bo=Points(bo_gd_east,bo_gd_north);

# ------------------------------------------------------------------------------#
# ------------------------------------------------------------------------------#

def setProblemData(inData: List) -> List:
    dataList=[];
    for row in inData:
        (gf_gebiet,gf_name,gf_geo_long,gf_geo_lat,gf_alti,gf_ort)=row;  # Gipfeldaten
        try:
            # Gipfelfußpunkt: Umrechnung Länge und Breite in UTM Koordinaten
            (gf_gd_east,gf_gd_north,gf_mdconv)=Lat_Long_To_Grid(gf_geo_lat,gf_geo_long,BO_Zone);
        except:
            print('!!! Unerwarteter Fehler beim Konvertieren - Abbruch');
            print('!!! Gipfelkoordinaten nicht nach UTM konvertierbar: ',gf_gebiet,gf_name);
            sys.exit();
        else:
            if gf_gd_east<100000 or gf_gd_east>900000:
                print('!!! Fehler - Abbruch');
                print('!!! Gipfel liegt außerhalb der UTM-Zone 32: ',gf_gebiet,gf_name);
                sys.exit();
            # end if
        # end try

        # ----------------------------------------------------------------------#

        # Gipfelfußpunkt in UTM Koordinaten
        pt_gd_gf=Points(gf_gd_east,gf_gd_north);

        # Vektor in UTM Koordinaten vom Beobachtungsfußpunkt zum Gipfelfußpunkt
        vc_gd_bo_gf=pt_gd_gf-pt_gd_bo;

        # Abstand (Länge in Metern) Beobachterfußpunkt - Gipfelfußpunkt
        # Auf Längenkorrekturen verzichte ich, sie spielen hier
        # bei den 'kurzen' Abständen und der gefragten Genauigkeit keine Rolle.
        ds_gd_bo_gf=vc_gd_bo_gf.lengthOf();  # in Längenmaß Meter

        # Azimutwinkel des Gipfelfußpunktes gegen UTM-Gitter Nord
        dirc=twopi-vc_gd_bo_gf.dirAngleOf()+halfpi;  # in Winkelmaß Radian

        # Azimutwinkel des Gipfelfußpunktes gegen geografisch Nord
        dirc=angles.addAngles(dirc,BO_MConv);  # in Winkelmaß Radian

        # ----------------------------------------------------------------------#

        # Kartesisches Koordinatensystem im Mittelpunkt der erweiterten
        # Gaußschen Kugelsphäre (Radius re_bo_NN)

        # Winkel Gipfelfußpunkt-Mittelpunkt-Beobachterfußpunkt
        # Gegeben durch die Kreisbogenlänge geteilt durch Radius
        theta=ds_gd_bo_gf/re_bo_NN;  # in Radian, 0<=theta<pi/2

        # Kartesische Koordinaten des Gipfelpunktes
        gf_kt_x=(re_bo_NN+gf_alti)*sin(theta);
        gf_kt_y=(re_bo_NN+gf_alti)*cos(theta);

        # Gipfelpunkt in kartesischen Kooedinaten
        pt_kt_gf=Points(gf_kt_x,gf_kt_y);

        # Vektor vom Beobachtungspunkt zum Gipfelpunkt
        vc_kt_bo_gf=pt_kt_gf-pt_kt_bo;

        # Sektor aus Gipfelpunkt - Beobachtungspunkt - Tangentenpunkt
        SecPTgPGf=Sectors(pt_kt_bo,vc_kt_bo_tg,vc_kt_bo_gf);

        # Orientierter Öffnungswinkel des Sektors
        elev=SecPTgPGf.openingAngleOf_oriented();
        # in Radian, es sollte gelten -pi<=gamma<pi
        elev=elev*rd2dg;  # in dezimalem Winkelgrad °

        # ----------------------------------------------------------------------#
        dataList.append((gf_name,pt_kt_gf,dirc,gf_mdconv,ds_gd_bo_gf,gf_alti,gf_gebiet,gf_ort,elev));
        # >>>>>>>>>>>>>    0        1      2      3           4           5        6     7      8
        # ----------------------------------------------------------------------#
    # end for

    dataList.sort(key=lambda listelem:listelem[2]);
    return dataList;

# end setProblemData

# ------------------------------------------------------------------------------#
# ------------------------------------------------------------------------------#

if __name__=='__main__':
    print('\n# Beginning alpenpanorama ...\n');

    # Ausgabe der Azimutwerte für den Kompass
    withCompasAzimuths: bool=True;
    angle_mag=0.0;
    (MagDevPossible,magdevValue)=getMagneticDeviation(BO_Missweisung);
    withCompasAzimuths=withCompasAzimuths and MagDevPossible;

    # Formatierte Datenausgabe auf die Konsole
    # --------------------------------------------------------------------------#

    BO_Jahr=str(datetime.date.today().year).strip();

    titel='Alpenpanorama '+BO_Jahr+' in Zahlen';
    bohdr1='Beobachtungsstandort';
    bohdr2='Länge; Breite; Höhe';
    bohdr3='Entfernung naut. Horizont';
    einheiten_str='Azimut, Elevation: Bogengrad; Entfernung: Kilometer; Höhe: Meter';
    bohdr4='Einheiten';

    nb=98;
    nbt=30;
    if withCompasAzimuths: nb=nb+8;

    print('\n');
    print(nb*'=');
    print(nbt*' ',titel);
    print(nb*'-');
    print('{0:28}: '.format(bohdr1),BO_Name);
    print('{0:28}: '.format(bohdr2),''+d2c(BO_String));
    print('{0:28}: '.format(bohdr3),d2c('{0:>4.1f}'.format(d_bo_tg/1000.0)),'km');
    print('{0:28}: '.format(bohdr4),einheiten_str);
    print(nb*'=');

    gipfeldots='{0:28}: ';
    gipfelhdr1='Gipfel / *Ort* / *Hütte*';
    gipfelhdr2a='Azimut    Elev.  Entfg. Höhe  Gebirgszug              bergnaher Ort';
    gipfelhdr2b='Azimut   Azimut   Elev.  Entfg. Höhe  Gebirgszug              bergnaher Ort';
    gipfelhdr3a='geo.N.    NHN           NHN';
    gipfelhdr3b='geo.N.   magn.N.  NHN           NHN';

    if withCompasAzimuths:
        print(gipfeldots.format(gipfelhdr1),gipfelhdr2b);
    else:
        print(gipfeldots.format(gipfelhdr1),gipfelhdr2a);
    # end if
    if withCompasAzimuths:
        print(gipfeldots.format(''),gipfelhdr3b);
    else:
        print(gipfeldots.format(''),gipfelhdr3a);
    # end if
    print(nb*'-');

    # --------------------------------------------------------------------------#
    inputData=getInputData(inputFile);
    problemData=setProblemData(inputData);
    # --------------------------------------------------------------------------#

    previous_angle=0.0;
    for data in problemData:
        # Läuft über alle Gipfel, Hütten, Orte
        # -------------------------# data:
        name=data[0];
        # <<<<<<<<<<<<<<<<<<<<<<<<<# 0 - name
        # -------------------------# 1 - pt - not used
        angle_geo=data[2]*rd2dg;
        # <<<<<<<<<<<<<<<<<<<<<<<<<# 2 - dirc
        # -------------------------# 3 - mdconv - not used
        dist_km=data[4]/1000.0;
        # <<<<<<<<<<<<<<<<<<<<<<<<<# 4 - dist
        alti_NN=data[5];
        # <<<<<<<<<<<<<<<<<<<<<<<<<# 5 - alti_NN
        gebiet=data[6];
        # <<<<<<<<<<<<<<<<<<<<<<<<<# 6 - gebiet
        ort=data[7];
        # <<<<<<<<<<<<<<<<<<<<<<<<<# 7 - ort
        elev=data[8];
        # <<<<<<<<<<<<<<<<<<<<<<<<<# 8 - elev, Bogengrad °
        # ------------------------#
        # ----------------------------------------------------------------------#

        isOrt=False;
        if gebiet=='': isOrt=True;
        if isOrt: name=' *'+name+'*';

        if withCompasAzimuths:
            # angle_mag=angle_geo+magdevValue;
            angle_mag=angles.addAngles(angle_geo,magdevValue,'deg');
        # end if

        if (previous_angle>0.0):
            base_angle=math.floor(previous_angle);
            last_angle=math.floor(angle_geo);
            delta_angle=last_angle-base_angle-1;
            if (delta_angle>0):
                for idx in range(1,delta_angle+1,1):
                    # ---------------------------------------------------------#
                    print('{0:.<28}:'.format(''),
                          d2c(' {0:.0f}  '.format(base_angle+idx)));
                    # ---------------------------------------------------------#
                # end for
            # end if
        # end if
        # ----------------------------------------------------------------------#

        pstr1='{0:.<28}:'.format(name);
        pstr2=d2c(' {0:.2f} '.format(angle_geo));

        if withCompasAzimuths:
            pstr3=d2c(' {0:.2f} '.format(angle_mag));
        else:
            pstr3='';
        # end if

        if isOrt:
            pstr4=' '+4*'.'+' ';
        elif elev<0.0:
            pstr4=' '+4*'|'+' ';
        else:
            pstr4=d2c(' {0:.2f} '.format(elev));
        # end if

        pstr5=d2c('{0:>6.1f}'.format(dist_km));
        pstr6=d2c(' {0:4d} '.format(alti_NN));
        pstr7='{0:.<23}'.format(gebiet);
        pstr8='{0:.<13}'.format(ort);
        # ----------------------------------------------------------------------#
        print(pstr1,pstr2,pstr3,pstr4,pstr5,pstr6,pstr7,pstr8);
        # ---------------------------------------------------------------------#
        previous_angle=angle_geo;
    # end for
    # --------------------------------------------------------------------------#
    print(nb*'-');

    if withCompasAzimuths:
        print(">>> Der Azimutwinkel gegen 'magn.N.' (Kompass Nord) gilt für das Beobachtungsjahr "+BO_Jahr+".");
    elif withCompasAzimuths and not MagDevPossible:
        print(">>> Es ist keine Nadelabweichung für das Beobachtungsjahr "+BO_Jahr+" verfügbar.");
        print(">>> Die Azimutwinkel gegen Kompass Nord können nicht ausgegeben werden.");
    # end if

    print(nb*'-');
    print("Die Höhenwinkel/Elevation der Gipfelpunkte beziehen sich auf den nautischen Horizont (0m NHN/müA).");
    print("Am Beobachtungsstandpunkt sind daher nur die relativen Werte der Höhenwinkel relevant!");
    print("Als Referenzwert für die Elevation wird man einen bekannten Gipfel, z.B. die Zugspitze, wählen.");
    print("Ist der berechnete Höhenwinkel für einen der Gipfel negativ, werden nur Striche |||| ausgegeben.");
    print("Für die eingestreuten Orte und Hütten werden statt der Höhenwinkel nur Punkte .... ausgegeben.");
    print(nb*'=');
    # --------------------------------------------------------------------------#
# end if
